const { app, BrowserWindow, ipcMain, dialog, shell } = require('electron');
const path = require('path');
const fs = require('fs');
const { spawn, exec } = require('child_process');

const isDev = !app.isPackaged;
const appDirectory = isDev ? app.getAppPath() : path.dirname(app.getPath('exe'));
const assetsRoot = isDev ? path.join(__dirname, '..') : process.resourcesPath;

const USER_DATA_FOLDER_PATH = path.join(appDirectory, 'UserData');
const SAVED_GAMES_FOLDER_PATH = path.join(USER_DATA_FOLDER_PATH, '../UserData/Saved Games');
const DOWNLOAD_QUEUE_FOLDER_PATH = path.join(USER_DATA_FOLDER_PATH, 'download_queue');
const USER_SETTINGS_FILE = path.join(USER_DATA_FOLDER_PATH, 'user_settings.json');
const FAVORITES_FOLDER_PATH = path.join(USER_DATA_FOLDER_PATH, 'favorites');
const SERVER_CONFIG_FILE = path.join(USER_DATA_FOLDER_PATH, 'server_config.json');

let mainWindow;
let pendingSavePrompts = {};

function getBaseGameName(filename) {
    const baseNameMatch = filename.match(/^([^()]+)/);
    return baseNameMatch ? baseNameMatch[1].trim() : filename.trim();
}

function createWindow() {
    app.commandLine.appendSwitch('high-dpi-support', '1');
    app.commandLine.appendSwitch('force-device-scale-factor', '1');
    mainWindow = new BrowserWindow({
        width: 1280,
        height: 850,
        webPreferences: {
            nodeIntegration: false,
            contextIsolation: true,
            preload: path.join(__dirname, 'preload.js')
        },
        backgroundColor: '#1e1e1e',
        autoHideMenuBar: true,
        fullscreen: true,
        show: false
    });
    if (isDev) {
        mainWindow.webContents.openDevTools();
    }
    mainWindow.maximize();
    
    if (isServerConfigValid()) {
        mainWindow.loadFile(path.join(__dirname, 'index.html'));
        mainWindow.once('ready-to-show', () => mainWindow.show());
    } else {
        createUrlPromptWindow();
    }
    
    mainWindow.on('closed', () => { mainWindow = null; });
}

function isServerConfigValid() {
    if (!fs.existsSync(SERVER_CONFIG_FILE)) {
        return false;
    }
    try {
        const configContent = fs.readFileSync(SERVER_CONFIG_FILE, 'utf-8');
        const config = JSON.parse(configContent);
        return config.ROM_Server_URL === "https://myrient.erista.me";
    } catch (error) {
        return false;
    }
}

function createUrlPromptWindow() {
    const urlPromptWindow = new BrowserWindow({
        width: 720,
        height: 300,
        webPreferences: {
            nodeIntegration: true,
            contextIsolation: false
        },
        backgroundColor: '#1e1e1e',
        autoHideMenuBar: true,
        resizable: false,
        parent: mainWindow,
        modal: true,
        show: false,
        title: "Server URL Required"
    });

    urlPromptWindow.loadFile(path.join(__dirname, 'serverURL.html'));
    urlPromptWindow.once('ready-to-show', () => {
        urlPromptWindow.show();
    });

    ipcMain.once('close-prompt-and-reload-main', () => {
        if (urlPromptWindow) {
            urlPromptWindow.close();
        }
        if (mainWindow) {
            mainWindow.loadFile(path.join(__dirname, 'index.html'));
            mainWindow.once('ready-to-show', () => mainWindow.show());
        }
    });
}

app.on('ready', () => {
    [USER_DATA_FOLDER_PATH, SAVED_GAMES_FOLDER_PATH, DOWNLOAD_QUEUE_FOLDER_PATH, FAVORITES_FOLDER_PATH].forEach(dir => {
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }
    });
    createWindow();
});

app.on('window-all-closed', () => { if (process.platform !== 'darwin') app.quit(); });
app.on('activate', () => { if (BrowserWindow.getAllWindows().length === 0) createWindow(); });

ipcMain.on('submit-server-url', (event, url) => {
    const cleanedUrl = url.trim().toLowerCase();
    const validUrls = ["https://myrient.erista.me", "myrient.erista.me", "erista.me"];

    if (validUrls.includes(cleanedUrl)) {
        const configData = { ROM_Server_URL: "https://myrient.erista.me" };
        fs.writeFileSync(SERVER_CONFIG_FILE, JSON.stringify(configData, null, 2));
        event.sender.send('url-accepted');
    } else {
        event.sender.send('serverURL-error', 'Invalid URL. Please try again.');
    }
});

ipcMain.handle('get-initial-data', async () => {
    try {
        const toFileUrl = (p) => 'file:///' + p.replace(/\\/g, '/');
        const consolesFolderPath = path.join(assetsRoot, 'data', 'Games');
        const consoleFolders = fs.existsSync(consolesFolderPath)
            ? fs.readdirSync(consolesFolderPath, { withFileTypes: true }).filter(d => d.isDirectory()).map(d => d.name)
            : [];

        const favoriteGames = {};
        consoleFolders.forEach(folder => {
            const favoritesFilePath = path.join(FAVORITES_FOLDER_PATH, `${folder}_favorites.txt`);
            if (fs.existsSync(favoritesFilePath)) {
                favoriteGames[folder] = fs.readFileSync(favoritesFilePath, 'utf-8').split('\n').filter(Boolean);
            } else {
                favoriteGames[folder] = [];
            }
        });

        const gamesByConsole = {};
        const arcadeTitles = {};

        for (const folder of consoleFolders) {
            const consolePath = path.join(consolesFolderPath, folder);
            let gameFiles = [];

            if (folder.startsWith('Arcade')) {
                const arcadeJsonPath = path.join(consolePath, 'arcade.json');
                if (fs.existsSync(arcadeJsonPath)) {
                    try {
                        const jsonContent = fs.readFileSync(arcadeJsonPath, 'utf-8');
                        const gamesData = JSON.parse(jsonContent);
                        if (Array.isArray(gamesData)) {
                            gamesData.forEach(game => {
                                if (game.rom_name && game.game_title) {
                                    gameFiles.push(game.rom_name);
                                    const baseName = getBaseGameName(path.parse(game.rom_name).name);
                                    arcadeTitles[baseName] = game.game_title;
                                }
                            });
                        }
                    } catch (error) {
                         console.error(`Error processing arcade.json:`, error);
                    }
                } else {
                    console.warn(`arcade.json not found for Arcade console.`);
                }
            } else {
                const jsonFilePath = path.join(consolePath, '_games.json');
                if (fs.existsSync(jsonFilePath)) {
                    try {
                        const jsonContent = fs.readFileSync(jsonFilePath, 'utf-8');
                        const gamesData = JSON.parse(jsonContent);
                        if (Array.isArray(gamesData)) {
                            gameFiles = gamesData.map(game => game.rom_name).filter(Boolean);
                        }
                    } catch (error) {
                        console.error(`Error reading or parsing _games.json for console '${folder}':`, error);
                    }
                } else {
                     console.warn(`_games.json not found for console '${folder}'. No games will be loaded.`);
                }
            }
            
            const gameGroups = {};
            gameFiles.forEach(file => {
                if (file.startsWith('.')) return;
                const filePath = path.join(consolePath, file);
                const filenameWithoutExt = path.parse(file).name;
                const baseGameName = getBaseGameName(filenameWithoutExt);
                if (!gameGroups[baseGameName]) {
                    gameGroups[baseGameName] = { baseName: baseGameName, versions: [] };
                }
                gameGroups[baseGameName].versions.push({ title: filenameWithoutExt, applicationPath: filePath });
            });

            const sortedGameGroups = Object.values(gameGroups);
            if (folder.startsWith('Arcade')) {
                sortedGameGroups.sort((a, b) => {
                    const titleA = arcadeTitles[a.baseName] || a.baseName;
                    const titleB = arcadeTitles[b.baseName] || b.baseName;
                    return titleA.localeCompare(titleB);
                });
            } else {
                sortedGameGroups.sort((a, b) => a.baseName.localeCompare(b.baseName));
            }
            gamesByConsole[folder] = sortedGameGroups;
        }

        return {
            consoleFolders,
            favoriteGames,
            gamesByConsole,
            arcadeTitles,
            paths: {
                imagesBasePath: path.join(assetsRoot, 'data', 'images'),
                iconsBasePathUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'icons')),
                defaultImageUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'default.png')),
                welcomeImageUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'welcome.png')),
                logoUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'logo.png')),
                scanlinesUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'scanlines.png')),
                buttonsUrl: toFileUrl(path.join(assetsRoot, 'data', 'images', 'buttons.png')),
                powerIcons: {
                    suspend: toFileUrl(path.join(assetsRoot, 'data', 'images', 'suspend.png')),
                    shutdown: toFileUrl(path.join(assetsRoot, 'data', 'images', 'off.png')),
                    restart: toFileUrl(path.join(assetsRoot, 'data', 'images', 'restart.png')),
                    exit: toFileUrl(path.join(assetsRoot, 'data', 'images', 'exit.png')),
                }
            }
        };
    } catch (error) {
        console.error("Fatal error getting initial data:", error);
        return null;
    }
});

ipcMain.handle('get-downloaded-games', (event, consoleName) => {
    if (!consoleName) return [];
    const consoleSavesPath = path.join(SAVED_GAMES_FOLDER_PATH, consoleName);
    if (!fs.existsSync(consoleSavesPath)) {
        return [];
    }
    try {
        const files = fs.readdirSync(consoleSavesPath);
        return files.map(file => {
            const stats = fs.statSync(path.join(consoleSavesPath, file));
            if (stats.isDirectory()) {
                return file;
            }
            return path.parse(file).name;
        }).map(name => getBaseGameName(name));
    } catch (error) {
        console.error(`Error reading saved games for ${consoleName}:`, error);
        return [];
    }
});

function resolveBackendScript(scriptName) {
    const backendPath = path.join(assetsRoot, 'backend');
    const pyPath = path.join(backendPath, `${scriptName}.py`);
    const exePath = path.join(backendPath, `${scriptName}.exe`);

    if (fs.existsSync(pyPath)) {
        return { executable: 'python', args: [pyPath] };
    }
    if (fs.existsSync(exePath)) {
        return { executable: exePath, args: [] };
    }
    throw new Error(`Backend script ${scriptName} not found in ${backendPath}`);
}

async function spawnEmulator(args) {
    try {
        const { executable, args: baseArgs } = resolveBackendScript('emu_app');
        const finalArgs = [...baseArgs, ...args];
        const child = spawn(executable, finalArgs, { detached: false, stdio: ['ignore', 'pipe', 'pipe'] });
        
        if (!child.pid) throw new Error("Failed to get PID from spawned process.");
        
        let stdoutBuffer = '';
        child.stdout.on('data', (data) => {
            stdoutBuffer += data.toString();
            let newlineIndex;
            while ((newlineIndex = stdoutBuffer.indexOf('\n')) >= 0) {
                const line = stdoutBuffer.substring(0, newlineIndex).trim();
                stdoutBuffer = stdoutBuffer.substring(newlineIndex + 1);
                if (line) {
                    try {
                        const progressData = JSON.parse(line);
                        if (progressData.type === 'prompt_save') {
                            pendingSavePrompts[child.pid] = { tempPath: progressData.temp_path, platform: progressData.platform };
                        } else {
                            mainWindow?.webContents.send('emu-progress-update', progressData);
                        }
                    } catch (e) {
                        mainWindow?.webContents.send('emu-log-message', `stdout: ${line}`);
                    }
                }
            }
        });
        child.stderr.on('data', (data) => mainWindow?.webContents.send('emu-log-message', `stderr: ${data.toString()}`));
        child.on('exit', (code, signal) => {
            mainWindow?.webContents.send('emu-process-exit', { pid: child.pid, code, signal });
            const promptData = pendingSavePrompts[child.pid];
            if (promptData) {
                mainWindow?.webContents.send('show-save-prompt', promptData);
                delete pendingSavePrompts[child.pid];
            }
        });
        child.on('error', (err) => { throw err; });

        return child.pid;
    } catch (error) {
        console.error(`Error spawning emulator: ${error}`);
        mainWindow?.webContents.send('emu-progress-update', { type: 'error', message: `Spawn failed: ${error.message}` });
        mainWindow?.webContents.send('emu-process-exit', { pid: null, code: -1, signal: null });
        throw error;
    }
}

ipcMain.handle('request-play-game', (e, gamePath) => spawnEmulator(['--mode', 'play', gamePath]));
ipcMain.handle('request-play-netplay-host', (e, gamePath) => {
    let username = 'Guest';
    if(fs.existsSync(USER_SETTINGS_FILE)) {
        username = JSON.parse(fs.readFileSync(USER_SETTINGS_FILE, 'utf-8')).netplay_username || 'Guest';
    }
    return spawnEmulator(['--mode', 'play-netplay-host', '--username', username, gamePath]);
});
ipcMain.handle('request-play-netplay-client', (e, opts) => {
    const consolePath = path.join(assetsRoot, 'data', 'Games', opts.consoleName);
    const gamePath = path.join(consolePath, `${opts.gameName}.zip`);
    
    let commandArgs = ['--mode', 'play-netplay-client', '--ip', opts.ip, '--port', opts.port];
    
    if (opts.mitm_session) {
        commandArgs.push('--mitm-ip', opts.mitm_ip);
        commandArgs.push('--mitm-session', opts.mitm_session);
    }
    
    commandArgs.push(gamePath);
    
    return spawnEmulator(commandArgs);
});
ipcMain.handle('request-save-game-app', (e, gamePath) => spawnEmulator(['--mode', 'save-app', gamePath]));
ipcMain.handle('request-save-game-external', (e, opts) => spawnEmulator(['--mode', 'save-external', '--output-path', opts.externalPath, opts.gamePath]));

ipcMain.on('kill-emu-app', (event, pid) => {
    if (pid) {
        try {
            if (pendingSavePrompts[pid]) delete pendingSavePrompts[pid];
            process.kill(pid, 'SIGTERM');
        } catch (error) {
            if (error.code === 'ESRCH') console.log(`Process ${pid} not found (already exited?).`);
            else console.error(`Error killing process ${pid}:`, error);
        }
    }
});

ipcMain.on('save-prompt-response', (event, { decision, tempPath, platform }) => {
    if (decision === 'yes') {
        spawnEmulator(['--mode', 'move-to-saved', '--temp-path', tempPath, '--platform', platform, '--output-path', SAVED_GAMES_FOLDER_PATH]);
    }
});

ipcMain.on('open-external-link', (event, url) => shell.openExternal(url));
ipcMain.handle('electron.dialog.showSaveDialog', (e, options) => dialog.showSaveDialog(mainWindow, options));
ipcMain.on('request-open-saved-games-folder', () => shell.openPath(SAVED_GAMES_FOLDER_PATH).catch(err => console.error("Could not open folder:", err)));
ipcMain.on('app-quit', () => app.quit());
ipcMain.on('suspend-pc', () => exec('powershell.exe -Command "Add-Type -Assembly System.Windows.Forms; [System.Windows.Forms.Application]::SetSuspendState(\'Suspend\', $false, $true)"'));
ipcMain.on('execute-shutdown', () => exec('shutdown /s /t 0'));
ipcMain.on('execute-restart', () => exec('shutdown /r /t 0'));

ipcMain.handle('get-username', () => {
    if (fs.existsSync(USER_SETTINGS_FILE)) {
        return JSON.parse(fs.readFileSync(USER_SETTINGS_FILE, 'utf-8')).netplay_username || null;
    }
    return null;
});

ipcMain.on('set-username', (event, username) => {
    let settings = {};
    if (fs.existsSync(USER_SETTINGS_FILE)) {
        settings = JSON.parse(fs.readFileSync(USER_SETTINGS_FILE, 'utf-8'));
    }
    settings.netplay_username = username;
    fs.writeFileSync(USER_SETTINGS_FILE, JSON.stringify(settings, null, 2));
});

ipcMain.on('toggle-favorite-game', (event, { gameBaseName, consoleFolderName }) => {
    const favoritesFilePath = path.join(FAVORITES_FOLDER_PATH, `${consoleFolderName}_favorites.txt`);
    let favorites = fs.existsSync(favoritesFilePath) ? fs.readFileSync(favoritesFilePath, 'utf-8').split('\n').filter(Boolean) : [];
    const index = favorites.indexOf(gameBaseName);
    if (index > -1) {
        favorites.splice(index, 1);
    } else {
        favorites.push(gameBaseName);
    }
    fs.writeFileSync(favoritesFilePath, favorites.join('\n'));
});

ipcMain.handle('request-lobby-list', async () => {
    const { executable, args } = resolveBackendScript('lobby');
    return new Promise((resolve, reject) => {
        const child = spawn(executable, args);
        let stdout = '';
        child.stdout.on('data', (data) => stdout += data);
        child.stderr.on('data', (data) => console.error(`Lobby stderr: ${data}`));
        child.on('error', reject);
        child.on('exit', (code) => {
            if (code === 0) {
                try { resolve(JSON.parse(stdout)); } catch(e) { reject(e); }
            } else {
                reject(new Error(`Lobby script exited with code ${code}`));
            }
        });
    });
});

ipcMain.handle('find-game-image', (e, { imageDir, gameTitle }) => {
    if (!fs.existsSync(imageDir)) return null;
    const exts = ['.png', '.jpg', '.jpeg', '.webp'];
    const files = fs.readdirSync(imageDir);
    const toFileUrl = (p) => 'file:///' + p.replace(/\\/g, '/');
    for (const file of files) {
        const ext = path.extname(file).toLowerCase();
        const base = path.parse(file).name;
        if (exts.includes(ext) && getBaseGameName(base) === getBaseGameName(gameTitle)) {
            return toFileUrl(path.join(imageDir, file));
        }
    }
    return null;
});

ipcMain.on('add-game-to-queue', (event, { consoleName, filename }) => {
    const queueFilePath = path.join(DOWNLOAD_QUEUE_FOLDER_PATH, `${consoleName}.txt`);
    fs.appendFileSync(queueFilePath, filename + '\n');
});

function getRetroArchConfigPath() {
    return path.join(assetsRoot, 'Emulators', 'RetroArch', 'retroarch.cfg');
}

ipcMain.handle('get-relay-config', () => {
    const cfgPath = getRetroArchConfigPath();
    const defaultConfig = { server: "", enabled: "false" };
    if (!fs.existsSync(cfgPath)) {
        return defaultConfig;
    }

    try {
        const content = fs.readFileSync(cfgPath, 'utf-8');
        const lines = content.split('\n');
        let config = { ...defaultConfig };
        
        const enabledLine = lines.find(line => line.trim().startsWith('netplay_use_mitm_server'));
        if (enabledLine) {
            const match = enabledLine.match(/"(true|false)"/);
            if (match) config.enabled = match[1];
        }

        const serverLine = lines.find(line => line.trim().startsWith('netplay_mitm_server'));
        if (serverLine) {
            const match = serverLine.match(/"([^"]+)"/);
            if (match) config.server = match[1];
        }
        
        return config;
    } catch (error) {
        console.error("Error reading retroarch.cfg:", error);
        return defaultConfig;
    }
});

ipcMain.on('set-relay-config', (event, { server, enabled }) => {
    const cfgPath = getRetroArchConfigPath();
    if (!fs.existsSync(cfgPath)) {
        console.error("retroarch.cfg not found, cannot set relay config.");
        return;
    }
    
    try {
        const content = fs.readFileSync(cfgPath, 'utf-8');
        let lines = content.split('\n');

        let enabledFound = false;
        let serverFound = false;

        lines = lines.map(line => {
            const trimmedLine = line.trim();
            if (trimmedLine.startsWith('netplay_use_mitm_server')) {
                enabledFound = true;
                return `netplay_use_mitm_server = "${enabled}"`;
            }
            if (trimmedLine.startsWith('netplay_mitm_server')) {
                serverFound = true;
                return `netplay_mitm_server = "${server}"`;
            }
            return line;
        });

        if (!enabledFound) {
            lines.push(`netplay_use_mitm_server = "${enabled}"`);
        }
        if (!serverFound) {
            lines.push(`netplay_mitm_server = "${server}"`);
        }

        fs.writeFileSync(cfgPath, lines.join('\n').replace(/\r/g, ''));
    } catch (error) {
        console.error("Error writing to retroarch.cfg:", error);
    }
});